// Flat shaded cube demo for the Saturn
// Mic, 2006

#include <stdlib.h>
#include "shared.h"
#include "sintb.h"

/* Command table elements */
#define CMDCTRL 0x00
#define CMDLINK 0x01
#define CMDPMOD 0x02
#define CMDCOLR 0x03
#define CMDSRCA 0x04
#define CMDSIZE 0x05
#define CMDXA   0x06
#define CMDYA   0x07
#define CMDXB   0x08
#define CMDYB   0x09
#define CMDXC   0x0A
#define CMDYC   0x0B
#define CMDXD   0x0C
#define CMDYD   0x0D
#define CMDGRDA 0x0E
#define CMDDUMY 0x0F

// Set the mesh type flag
#define MESH(n) (mesh_flag=(n)?0x0100:0x0000)

// Convert between integer and fixed point
#define INT2FIX(a) (((int)(a))<<10)
#define FIX2INT(a) (((int)(a))>>10)

/* Create a 5:5:5 RGB color with the MSB set */
#define COLOR(r,g,b)    (((r)&0x1F)|((g)&0x1F)<<5|((b)&0x1F)<<10|0x8000)


typedef struct {
	int x,y,z;
} point3d;

typedef struct {
	int p0,p1,p2,p3,color;
} quadrangle;


int 		list_index = 1;
uint16 		mesh_flag = 0;
int 		theta = 0;

// Points defining the cube
int 		points[8][3] = {{-1,-1,-1},{1,-1,-1},{1,1,-1},{-1,1,-1},
 						  {-1,-1,1},{1,-1,1},{1,1,1},{-1,1,1}};

// Faces defining the cube. The 5th element is a color mask.
int 		faces[6][5] = {{0,1,2,3,0x0000ff},{4,5,6,7,0x00ff00},{0,1,5,4,0xff0000},
    		               {5,1,2,6,0x00ffff},{3,2,6,7,0xff00ff},{0,4,7,3,0xffffff}};

point3d		*model=(point3d*)&points,rotated[8],projected[8],camera;
quadrangle 	*cube=(quadrangle*)&faces;
int 		faceOrder[6];


/* Clear command list */
void list_clear(void) {
    uint16 *p = (uint16 *)VDP1_VRAM;
    int i;
    for(i=0;i<16;i++)
        p[i] = 0xFFFF;
    p[CMDCTRL] = 0x4000;
    list_index = 1;
}

/* Add end marker to command list */
void list_end(void) {
    uint16 *p = (uint16 *)(VDP1_VRAM + (list_index << 5));
    p[CMDCTRL] = 0x8000;
    list_index++;
}

/* Add system clip to command list */
void list_set_system_clip(int16 x, int16 y) {
    uint16 *p = (uint16 *)(VDP1_VRAM + (list_index << 5));

    p[CMDCTRL] = 0x0009; /* Set system clip */
    p[CMDLINK] = 0x0000; /* Jump address */
    p[CMDXC]   = x;
    p[CMDYC]   = y;
    list_index++;
}


/* Add user clip to command list */
void list_set_user_clip(int16 x1, int16 y1, int16 x2, int16 y2) {
    uint16 *p = (uint16 *)(VDP1_VRAM + (list_index << 5));

    p[CMDCTRL] = 0x0008; /* Set user clip */
    p[CMDLINK] = 0x0000; /* Jump address */
    p[CMDXA]   = x1;
    p[CMDYA]   = y1;
    p[CMDXC]   = x2;
    p[CMDYC]   = y2;
    list_index++;
}

/* Add local coords to command list */
void list_set_local_coords(int16 x, int16 y) {
    uint16 *p = (uint16 *)(VDP1_VRAM + (list_index << 5));

    p[CMDCTRL] = 0x000A; /* Set local coords */
    p[CMDLINK] = 0x0000; /* Jump address */
    p[CMDXA]   = x;
    p[CMDYA]   = y;
    list_index++;
}



/* Add local coords to command list */
void list_polygon(int16 x1, int16 y1, int16 x2, int16 y2, int16 x3, int16 y3, int16 x4, int16 y4, uint16 color, uint32 grda) {
    uint16 *p = (uint16 *)(VDP1_VRAM + (list_index << 5));

    p[CMDCTRL] = 0x0004; /* Polygon */
    p[CMDLINK] = 0x0000; /* Jump address */
    p[CMDPMOD] = 0x00C0 | mesh_flag | 0; //4;
    p[CMDCOLR] = color;
    p[CMDXA]   = x1;
    p[CMDYA]   = y1;
    p[CMDXB]   = x2;
    p[CMDYB]   = y2;
    p[CMDXC]   = x3;
    p[CMDYC]   = y3;
    p[CMDXD]   = x4;
    p[CMDYD]   = y4;
    p[CMDGRDA] = grda>>3;
    list_index++;
}


void rotate(point3d *in,point3d *out,int angle,int n) {
	int i;
	int xr,yr,zr,temp;
	int can,san;

	san = sintb[angle&0xff];
	can = sintb[(angle+0x40)&0xff];

	for (i=0; i<n; i++)
	{
		// About X
		out[i].y = ((in[i].y * can) - (in[i].z * san)); out[i].y=FIX2INT(out[i].y);
		out[i].z = ((in[i].y * san) + (in[i].z * can)); out[i].z=FIX2INT(out[i].z);

		// About Y
		out[i].x = ((in[i].x * can) - (out[i].z * san)); out[i].x=FIX2INT(out[i].x);
		out[i].z = ((in[i].x * san) + (out[i].z * can)); out[i].z=FIX2INT(out[i].z);

		// About Z
		temp = out[i].x;
		out[i].x = ((out[i].x * can) - (out[i].y * san)); out[i].x=FIX2INT(out[i].x);
		out[i].y = ((temp * san) + (out[i].y * can)); out[i].y=FIX2INT(out[i].y);
	}

}


void project(point3d *in,point3d *out,int n) {
	int i;

	for (i=0; i<n; i++) {
		out[i].x = camera.x + FIX2INT(((in[i].x * camera.z) / (camera.z - FIX2INT(in[i].z))));
		out[i].y = camera.y + FIX2INT(((in[i].y * camera.z) / (camera.z - FIX2INT(in[i].z))));
		out[i].z = in[i].z;
	}
}


int avgZ[32];	// Allow max 32 faces to be sorted
void sort_quads(quadrangle *f,point3d *p,int *order,int n) {
	int i,j,temp;

	// Initialize arrays
	for (i=0; i<n; i++)	{
		avgZ[i] = p[f[i].p0].z +
                  p[f[i].p1].z +
                  p[f[i].p2].z +
                  p[f[i].p3].z;
		order[i] = i;
	}


	// Bubble-sort the whole lot.. yeehaw!
	for (i=0; i<n-1; i++) {
		for (j=i+1; j<n; j++) {
			if (avgZ[j] < avgZ[i]) {
				temp = avgZ[i]; avgZ[i] = avgZ[j]; avgZ[j] = temp;
				temp = order[i]; order[i] = order[j]; order[j] = temp;
			}
		}
	}
}




int main() {
	int i,j,k,m;
	uint16 *p;

    vdp_init();
    conio_init();
    //pad_init();

    /* Set up VDP1 to appear under VDP2 graphics (0=hide, 1-7=visible) */
    PRISA = 0x0101;
    PRISB = 0x0101;
    PRISC = 0x0101;
    PRISD = 0x0101;

    /* Set up VDP1 registers */
    TVMR = 0x0000;
    FBCR = 0x0000;
    PTMR = 0x0002;
    EWDR = COLOR(7,7,20);    /* Erase color: gray */
    EWLR = 0x0000;          /* Erase range: (0,0) -> (352,240) */
    EWRR = 0x50DF;

	for (i=0; i<8; i++)
	{
		model[i].x = INT2FIX(model[i].x * 35);
		model[i].y = INT2FIX(model[i].y * 35);
		model[i].z = INT2FIX(model[i].z * 35);
	}

	camera.x = 160; camera.y = 112; camera.z = -200;

    /* Screen on */
    TVMD = 0x8000;

   /* Make Gouraud shading table */
    p = (uint16 *)0x25C60000;
    p[0] = COLOR(31,0,0);
    p[1] = COLOR(0,31,0);
    p[2] = COLOR(0,0,31);
    p[3] = COLOR(31,31,31);

    list_clear();
    list_end();

    /* Main loop */
    for(;;)
    {
        int i;

        wait_vblank_out();
        wait_vblank_in();

		TVMD = 0x8000;

      	/* Make new list */
        list_clear();
        list_set_system_clip(320, 224);
        list_set_user_clip(0, 0, 320, 224);
        list_set_local_coords(0, 0);

		rotate(model, rotated, theta++, 8);
		project(rotated, projected, 8);
		sort_quads(cube, rotated, faceOrder, 6);

        MESH(1);
 		for (i=0; i<6; i++)
 		{
			j = faceOrder[5-i];
			k = (((-FIX2INT(avgZ[5-i])/4)+35)*31)/70;	// Calculate shade, 0..31
			m = cube[j].color;							// Get color mask for this face

 			list_polygon( \
            projected[cube[j].p0].x,projected[cube[j].p0].y, \
            projected[cube[j].p3].x,projected[cube[j].p3].y, \
            projected[cube[j].p2].x,projected[cube[j].p2].y, \
            projected[cube[j].p1].x,projected[cube[j].p1].y, \
            COLOR(k&m,k&(m>>8),k&(m>>16)),
            0);
		}

        list_end();
     }

    vdp_shutdown();

	return 0;
}
