//----------------------------------------------------------------------------------------------------------------------
// Cruzer's library of commonly used stuff
//----------------------------------------------------------------------------------------------------------------------
// Tab size: 8
// Assembler: KickAssembler
//----------------------------------------------------------------------------------------------------------------------
#importonce
//----------------------------------------------------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------------------------------------------------
.const t = true
.const f = false
//----------------------------------------------------------------------------------------------------------------------
.const BEQ = BEQ_REL
.const BNE = BNE_REL
.const BCS = BCS_REL
.const BCC = BCC_REL
.const BPL = BPL_REL
.const BMI = BMI_REL
.const BVS = BVS_REL
.const BVC = BVC_REL
.const JSR = JSR_ABS
.const JMP = JMP_ABS
.const JAM = $02

.const LDA_IZY = LDA_IZPY
.const STA_IZY = STA_IZPY
.const ADC_IZY = ADC_IZPY
.const SBC_IZY = SBC_IZPY
.const ORA_IZY = ORA_IZPY
.const AND_IZY = AND_IZPY
.const EOR_IZY = EOR_IZPY
.const CMP_IZY = CMP_IZPY
.const LAX_IZY = LAX_IZPY

.const LDA_IZX = LDA_IZPX
.const STA_IZX = STA_IZPX
.const ADC_IZX = ADC_IZPX
.const SBC_IZX = SBC_IZPX
.const ORA_IZX = ORA_IZPX
.const AND_IZX = AND_IZPX
.const EOR_IZX = EOR_IZPX
.const CMP_IZX = CMP_IZPX
.const LAX_IZX = LAX_IZPX
.const SAX_IZX = SAX_IZPX

.const LDA_ABX = LDA_ABSX
.const STA_ABX = STA_ABSX
.const ADC_ABX = ADC_ABSX
.const SBC_ABX = SBC_ABSX
.const ORA_ABX = ORA_ABSX
.const AND_ABX = AND_ABSX
.const EOR_ABX = EOR_ABSX
.const CMP_ABX = CMP_ABSX
.const INC_ABX = INC_ABSX
.const DEC_ABX = DEC_ABSX
.const LDY_ABX = LDY_ABSX

.const LDA_ABY = LDA_ABSY
.const STA_ABY = STA_ABSY
.const ADC_ABY = ADC_ABSY
.const SBC_ABY = SBC_ABSY
.const ORA_ABY = ORA_ABSY
.const AND_ABY = AND_ABSY
.const EOR_ABY = EOR_ABSY
.const CMP_ABY = CMP_ABSY
.const LDX_ABY = LDX_ABSY

.const KOALA_P00 = "Bitmap = $001c, ScreenRam = $1f5c, ColorRam = $2344, BackgroundColor = $272c"
//----------------------------------------------------------------------------------------------------------------------
.const CIA1InterruptControl =	$dc0d
.const CIA1TimerAControl =	$dc0e
.const CIA1TimerBControl =	$dc0f
.const CIA2InterruptControl =	$dd0d
.const CIA2TimerAControl =	$dd0e
.const CIA2TimerBControl =	$dd0f
.const VIC2InteruptControl =	$d01a
.const VIC2InteruptStatus =	$d019
.const ProcessorPortDDRDefault =	%00101111
.const ProcessorPortAllRAMWithIO =	%00100101
//----------------------------------------------------------------------------------------------------------------------
// Misc
//----------------------------------------------------------------------------------------------------------------------
.pseudocommand jam {
	.by JAM
}

.macro basic(adr) {
	.pc = $0801 "basic"
	:BasicUpstart(adr)
}

.function _16bit_nextArgument(arg) {
	.if (arg.getType() == AT_IMMEDIATE) {
		.return CmdArgument(arg.getType(), >arg.getValue())
	}
	.return CmdArgument(arg.getType(), arg.getValue() + 1)
}

.macro avoidPageBreaks(numBytes) {
	:avoidPageBreaksWithJump(numBytes, f)
}

.macro avoidPageBreaksWithJump(numBytes, jump) {
	.if (numBytes > 255) {
		.error "Can't avoid page breaks for more than 255 bytes"
	}
	.var currentPage = floor(* / 256)
	.var nextPage = floor((* + numBytes) / 256)
	.if (currentPage != nextPage) {
		.if (jump) {
			jmp !+
		}
		.eval currentPage = floor(* / 256)
		.eval nextPage = floor((* + numBytes) / 256)
		.if (currentPage != nextPage) {
			.align $100
		}
		!:
	}
}
//----------------------------------------------------------------------------------------------------------------------
// Move & stuff
//----------------------------------------------------------------------------------------------------------------------

// Move byte
//
.pseudocommand mb src : tar {
	lda src
	sta tar
}

// Move byte with x
//
.pseudocommand mbx src : tar {
	ldx src
	stx tar
}

// Move byte with y
//
.pseudocommand mby src : tar {
	ldy src
	sty tar
}

// Move word
//
.pseudocommand mw src : tar {
	
	//TODO: doesn't work: with (zp,x)
	
	lda src
	sta tar

	.var yInced = false
	
	.if (src.getType() == AT_IZEROPAGEY) {
		iny
		lda src
		.eval yInced = true
	} else {
		lda _16bit_nextArgument(src)
	}

	.if (tar.getType() == AT_IZEROPAGEY) {
		.if (!yInced) iny
		sta tar
	} else {
		sta _16bit_nextArgument(tar)
	}
}

// Move word
//
.pseudocommand mow arg1 : arg2 {
	:mw arg1 : arg2
}

// Store word
//
.pseudocommand sw tar {	
	sta tar
	sta _16bit_nextArgument(tar)
}

// Load word into x/y
//
.pseudocommand lxy src {
	ldx src
	ldy _16bit_nextArgument(src)
}

// Store word from x/y
//
.pseudocommand sxy tar {	
	stx tar
	sty _16bit_nextArgument(tar)
}

// Add byte
//
.pseudocommand ab src:tar {
	clc
	lda src
	adc tar
	sta tar
}

// Add word
//
.pseudocommand aw adr:val {
	lda adr
	clc
	adc val
	sta adr
	lda _16bit_nextArgument(adr)
	adc _16bit_nextArgument(val)
	sta _16bit_nextArgument(adr)
}


// inc word
//
.pseudocommand iw arg {{
	inc arg
	bne !+
	inc _16bit_nextArgument(arg)
!:
}}


// dec word
//
.pseudocommand dw arg {
	lda arg
	bne !+
	dec _16bit_nextArgument(arg)
!:
	dec arg
}

//----------------------------------------------------------------------------------------------------------------------
// Lists
//----------------------------------------------------------------------------------------------------------------------
.macro dump(list) {
	.fill list.size(), list.get(i)
}

.macro dumpLo(list) {
	.fill list.size(), <list.get(i)
}

.macro dumpHi(list) {
	.fill list.size(), >list.get(i)
}

.macro dumpWo(list) {
	.for (var i=0; i<list.size(); i++) {
		.wo list.get(i)
	}
}

.macro dump2Dlist(list) {
	.for (var y=0; y<list.size(); y++) {
		.var xList = list.get(y)
		.fill xList.size(), xList.get(i)
	}
}

// Create List filled with value
//
.function list(size, value) {
	.var list = List()
	.for (var i=0; i<size; i++) {
		.eval list.add(value)
	}
	.return list
}

.function list(size) {
	.return list(size, 0)
}

.function list2D(size0, size1) {
	.var list = List()
	.for (var i=0; i<size0; i++) {
		.eval list.add(list(size1))
	}
	.return list
}

.function list3D(size0, size1, size2) {
	.var list = List()
	.for (var i=0; i<size0; i++) {
		.eval list.add(list2D(size1, size2))
	}
	.return list
}

.function list4D(size0, size1, size2, size3) {
	.var list = List()
	.for (var i=0; i<size0; i++) {
		.eval list.add(list3D(size1, size2, size3))
	}
	.return list
}

.function removeDuplicates(srcList) {
	.var res = List()
	.var index = Map()
	.for (var i=0; i<srcList.size(); i++) {
		.var element = srcList.get(i)
		.if (index.get(element) == null) {
			.eval res.add(element)
			.eval index.put(element, t)
		}
	}
	.return res
}
//----------------------------------------------------------------------------------------------------------------------
// HashMaps
//----------------------------------------------------------------------------------------------------------------------
.function Map() {
	.return Hashtable()
}

.function toMap(list) {
	.var map = Map()
	.for (var i=0; i<list.size(); i++) {
		.eval map.put(list.get(i), t)
	}
	.return map
}

.function list2hash(list) {
	.return toMap(list)
}

.function toIndexedMap(list) {
	.var map = Map()
	.for (var i=0; i<list.size(); i++) {
		.eval map.put(list.get(i), i)
	}
	.return map
}
//----------------------------------------------------------------------------------------------------------------------
// Color and other nybble manipulation
//----------------------------------------------------------------------------------------------------------------------

.var charToColor = Map()

.for (var i=0; i<16; i++) {
	.eval charToColor.put(hex(i), i)
}

.eval charToColor.put(' ', 0)
.eval charToColor.put('-', 0)

.function toByte(c0, c1) {
	.return charToColor.get(c0) * 16 + charToColor.get(c1)
}

.function toWord(c0, c1, c2, c3) {
	.return toByte(c0, c1) * 256 + toByte(c2, c3)
}

.function toBytes(str) {
	.var bytes = List()
	.for (var i=0; i<str.size()-1; i++) {
		.if (str.charAt(i) != ' ' && str.charAt(i + 1) != ' ') {
			.eval bytes.add(toByte(str.charAt(i + 0), str.charAt(i + 1)))
		}
	}
	.return bytes
}


.function toWords(str) {
	.var words = List()
	.for (var i=0; i<str.size()-3; i++) {
		.var c0 = str.charAt(i + 0)
		.var c1 = str.charAt(i + 1)
		.var c2 = str.charAt(i + 2)
		.var c3 = str.charAt(i + 3)
		.if (c0 != ' ' && c1 != ' ' && c2 != ' ' && c3 != ' ') {
			.eval words.add(toWord(c0, c1, c2, c3))
		}
	}
	.return words
}

.macro dumpBytes(str) {
	.var l = toBytes(str)
	:dump(l)
}

.function toColorNybbles(str) {
	.var colorBytes = toColorBytes(str)
	.var list = List()
	.for (var i=0; i<colorBytes.size(); i=i+2) {
		.var color0 = colorBytes.get(i)
		.var color1 = 0
		.if (colorBytes.size() > i+1) {
			.eval color1 = colorBytes.get(i + 1)
		}
		.var byte = color0 << 4 | color1
		.eval list.add(byte)
	}
	.return list
}

.function toColorNybbles_OLD(str) {
	.var list = List()
	.for (var i=0; i<str.size(); i=i+2) {
		.var chr0 = str.charAt(i)
		.var color0 = charToColor.get(chr0)
		.var color1 = 0
		.if (str.size() > i+1) {
			.var chr1 = str.charAt(i+1)
			.eval color1 = charToColor.get(chr1)
		}
		.var byte = color0 << 4 | color1
		.eval list.add(byte)
	}
	.return list
}

.function toColors(str) {
	.return toColorBytes(str)
}

.function toColorBytes(str) {
	.var list = List()
	.for (var i=0; i<str.size(); i++) {
		.var chr = str.charAt(i)
		.var color = charToColor.get(chr)
		.if (chr != " ") {
			.eval list.add(color)
		}
	}
	.return list
}

.macro dumpColorNybbles(str) {
	.var colz = toColorNybbles(str)
	.fill colz.size(), colz.get(i)
}

.macro dumpNybs(str) {
	:dumpColorNybbles(str)
}
.macro dumpNybsBackw(str) {
	.eval str = reverse(str)
	:dumpNybs(str)
}

.macro dumpColorBytes(str, startWithSize) {
	.if (startWithSize) {
		.by str.size()
	}
	.var colz = toColorBytes(str)
	.fill colz.size(), colz.get(i)
}

.macro colorBytes(str) {
	:dumpColorBytes(str, f)
}

.macro colorBytesLo(str) {
	.var colz = toColorBytes(str)
	.fill colz.size(), colz.get(i)
}
.macro colorBytesHi(str) {
	.var colz = toColorBytes(str)
	.fill colz.size(), colz.get(i) << 4
}

// Combine two different color lists into one by using lo/hi nybbles
//
.macro comboColors(str0, str1) {
	.var list = List()
	.for (var i=0; i<str0.size(); i++) {
		.var chr0 = str0.charAt(i)
		.var chr1 = str1.charAt(i)
		.var color0 = charToColor.get(chr0)
		.var color1 = charToColor.get(chr1)
		.var color = (color0 << 4) | color1
		.eval list.add(color)
	}
	:dump(list)
}
//----------------------------------------------------------------------------------------------------------------------
// Strings
//----------------------------------------------------------------------------------------------------------------------
.function reverse(str) {
	.var reverse = ""
	.for (var i=str.size()-1; i>=0; i--) {
		.eval reverse = reverse + str.charAt(i)
	}
	.return reverse
}

.function int(n) {
	.var str = "" + floor(n)
	.return str
}

.function int(n, digits) {
	.var str = "" + floor(n)
	.if (str.charAt(0) == 'N') .return "NaN"
	.for (;str.size()<digits;) {
		.eval str="0"+str
	}
	.return str
}

.function hex(n) {
	.return toHexString(n)
}

.function hex(n, digits) {
	.var str = toHexString(n)
	.if (str.charAt(0) == 'N') .return "NaN"
	.if (n >= 0) {
		.for ( ;str.size() < digits; str="0"+str){}
	} else {
	//	.print "before: $" + str
		.eval str = str.substring(str.size()-digits, str.size())
	//	.print "after:  $" + str
	}
	.return str
}

.function bin(n) {
	.return toBinaryString(n)
}

.function bin(n, digits) {
	.var str = toBinaryString(n)
	.if (n >= 0) {
		.for (; str.size() < digits; str = "0" + str) {}
	} else {
		.eval str = str.substring(str.size() - digits, str.size())
	}
	.return str
}


.function prettyFloat(n) {
	.if (abs(n) < 0.01 && abs(n) > 0) {
		.var pretty = prettyFloat(abs(n) + 1)
		.var sign = n < 0 ? "-" : " "
		.eval pretty = sign + "0" + pretty.substring(2, 7)
		.return pretty
	}
	.var rounded = floor(n * 10000) / 10000
	.var pretty = "" + rounded
	.if (pretty.charAt(0) != "-") {
		.eval pretty = " " + pretty
	}
	.var size = pretty.size()
	.var dotPos = -1
	.for (var i=0; i<size; i++) {
		.if (pretty.charAt(i) == '.') .eval dotPos = i
	}
	.if (dotPos == -1) {
		.eval pretty = pretty + ".0000"
	} else .if (size - dotPos < 5) {
		.for (var i=0; i<5-(size-dotPos); i++) {
			.eval pretty = pretty + "0"
		}
	}
	.return pretty
}

/*
.print "-99999:        " + prettyFloat(-99999)
.print "-1    :        " + prettyFloat(-1)
.print "-0.0001:       " + prettyFloat(-0.0001)
.print "0.0000:        " + prettyFloat(0)
.print "0.0001:        " + prettyFloat(0.0001)
.print "1.0001:        " + prettyFloat(1.0001)
.print "1000000:       " + prettyFloat(1000000)
*/


.function intListToString(intList) {
	.var result = ""
	.for (var i=0; i<intList.size(); i++) {
		.eval result = result + toIntString(intList.get(i))
		.if (i < intList.size() - 1) .eval result = result + ", "
	}
	.return result
}

.macro printList(prefix, list) {
	.var out = prefix
	.for (var i=0; i<list.size(); i++) {
		.eval out = out + toIntString(list.get(i))
		.if (i < list.size() - 1) .eval out = out + ", "
	}
	.print out
}

.macro dumpAndPrint(prefix, list, suffix) {
	dump(list)
	printHexList(prefix, list, suffix)
}

.macro printHexList(prefix, list, suffix) {
	.var max = 0
	.for (var i=0; i<list.size(); i++) .eval max = max(max, list.get(i))
	.var digits = 2
	.if (max >= $100) .eval digits = 4
	.var out = prefix
	.for (var i=0; i<list.size(); i++) {
		.eval out = out + "$" + hex(list.get(i), digits)
		.if (i < list.size() - 1) .eval out = out + ", "
	}
	.print out + suffix
}

.macro printIntList(prefix, list, suffix) {
	.var out = prefix
	.for (var i=0; i<list.size(); i++) {
		.eval out = out + int(list.get(i))
		.if (i < list.size() - 1) .eval out = out + ", "
	}
	.print out + suffix
}

.function printHexList(list) {
	.var max = 0
	.for (var i=0; i<list.size(); i++) {
		.eval max = max(max, list.get(i))
	}
	.var digits = 2
	.if (max >= $100) .eval digits = 4
	.var out = ""
	.for (var i=0; i<list.size(); i++) {
		.eval out = out + "" + hex(list.get(i), digits)
		.if (i < list.size() - 1) .eval out = out + " "
	}
	.print out
}

.macro printStringList(prefix, list) {
	.var out = prefix
	.for (var i=0; i<list.size(); i++) {
		.eval out = out + list.get(i)
		.if (i < list.size() - 1) .eval out = out + ", "
	}
	.print out
}

.function listToHexString(list) {
	.var str = ""
	.for (var i=0; i<list.size(); i++) {
		.eval str += hex(list.get(i), 2)
		.if (i < list.size()-1) .eval str += " "
	}
	.return str
}

.function padString(str, minLen) {
	.while (str.size() < minLen) {
		.eval str += " "
	}
	.return str
}
//----------------------------------------------------------------------------------------------------------------------
// Math
//----------------------------------------------------------------------------------------------------------------------

// get a random int which is >=min and <=max...
//
.function rnd(min, max) {
	.var range = max - min + 1
	.return floor(random() * range) + min
}

// get a random int which is not x
//
.function rndNot(min, max, x) {
	.var result = rnd(min, max)
	.if (result == x) {
		.eval result = rndNot(min, max, x)
	}
	.return result
}


// get a random int which is not 0...
//
.function rndNot0(min, max) {
	.return rndNot(min, max, 0)
}

// get a List with length len containing random ints which are >=min and <=max...
//
.function rndList(len, min, max) {
	.var result = List()
	.for (var i=0; i<len; i++) {
		.eval result.add(rnd(min,max))
	}
	.return result
}

// Binary logarithm
// E.g. which bit is set in the number 8? Answer: log2(8) = 3
//
.function log2(n) {
	.return log(n) / log(2)
}

// seems to be pretty much the same as log2
//
.function getBitNo(num) {
	.if (num <= 0) .error "getBitNo: " + num
	.var cnt = 0
	.for (var b=1; t; b=b<<1) {
		.if ((num & b) != 0) {
			.if ((num & b) == num) {
				.return cnt
			} else {
				.error "getBitNo: " + num
			}
		}
		.eval cnt++
	}
}

.macro shiftLeft(num) {
	.for (var i=0; i<num; i++) {
		asl
	}
}

.macro mul(num) {
	:shiftLeft(getBitNo(num))
}

.function sine(len, amp, min, repeat) {
	.var list = List()
	.for (var i=0; i<len; i++) {
		.var val = min + amp * 0.5 + amp * 0.499999 * sin(i / (len / 2 / PI))
		.eval list.add(val)
	}
	.if (repeat) {
		.eval list.addAll(list)
	}
	.return list
}

.macro sine(len, amp, min, repeat) {
	.var list = sine(len, amp, min, repeat)
	:dump(list)
}

.macro stdSine(amp) {
	:sine($100, amp, 0, t)
}
//----------------------------------------------------------------------------------------------------------------------
// Long branching
//----------------------------------------------------------------------------------------------------------------------
// Caused weird bugs sometimes - disabled
// TODO: Retest

/*
	.pseudocommand bpl tar {{
		.var dist = abs(tar.getValue() - *)
		bmi !+
		jmp tar
	!:
	}}
	
	.pseudocommand bmi tar {{
		.var dist = abs(tar.getValue() - *)
		bpl !+
		jmp tar
	!:
	}}
	
	.pseudocommand bne tar {{
		//.var dist = abs(tar.getValue() - *)
	
		beq !+
		jmp tar
	!:
	}}
	
	.pseudocommand bne2 tar {{
		//.var dist = abs(tar.getValue() - *)
	
		beq !+
		jmp tar
	!:
	}}
	
	.pseudocommand beq tar {{
		.var dist = abs(tar.getValue() - *)
	
		bne !+
		jmp tar
	!:
	}}

*/
//----------------------------------------------------------------------------------------------------------------------
// Interrupts
//----------------------------------------------------------------------------------------------------------------------
.macro setRasterLine(rasterline, d011) {
	:mb #<rasterline : $d012
	:mb #((rasterline & $100) >> 1) | d011 : $d011
}
//----------------------------------------------------------------------------------------------------------------------
