"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
var _semver = _interopRequireDefault(require("semver"));
var _getVersions = require("../../api/getVersions");
var _settings = require("../../api/settings");
var Log = _interopRequireWildcard(require("../../log"));
var _downloadExpoGoAsync = require("../../utils/downloadExpoGoAsync");
var _errors = require("../../utils/errors");
var _ora = require("../../utils/ora");
var _prompts = require("../../utils/prompts");
function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _interopRequireWildcard(obj) {
    if (obj && obj.__esModule) {
        return obj;
    } else {
        var newObj = {};
        if (obj != null) {
            for(var key in obj){
                if (Object.prototype.hasOwnProperty.call(obj, key)) {
                    var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {};
                    if (desc.get || desc.set) {
                        Object.defineProperty(newObj, key, desc);
                    } else {
                        newObj[key] = obj[key];
                    }
                }
            }
        }
        newObj.default = obj;
        return newObj;
    }
}
const debug = require("debug")("expo:utils:ExpoGoInstaller");
class ExpoGoInstaller {
    // Keep a list of [platform-deviceId] so we can prevent asking multiple times if a user wants to upgrade.
    // This can prevent annoying interactions when they don't want to upgrade for whatever reason.
    static cache = {};
    constructor(platform, appId, sdkVersion){
        this.platform = platform;
        this.appId = appId;
        this.sdkVersion = sdkVersion;
    }
    /** Returns true if the installed app matching the previously provided `appId` is outdated. */ async isClientOutdatedAsync(device) {
        const installedVersion = await device.getAppVersionAsync(this.appId);
        if (!installedVersion) {
            return true;
        }
        const version = await this._getExpectedClientVersionAsync();
        debug(`Expected Expo Go version: ${version}, installed version: ${installedVersion}`);
        return version ? !_semver.default.eq(installedVersion, version) : true;
    }
    /** Returns the expected version of Expo Go given the project SDK Version. Exposed for testing. */ async _getExpectedClientVersionAsync() {
        var ref, ref1;
        const versions = await (0, _getVersions).getVersionsAsync();
        // Like `sdkVersions['44.0.0']['androidClientVersion'] = '1.0.0'`
        const specificVersion = versions == null ? void 0 : (ref = versions.sdkVersions) == null ? void 0 : (ref1 = ref[this.sdkVersion]) == null ? void 0 : ref1[`${this.platform}ClientVersion`];
        const latestVersion = versions[`${this.platform}Version`];
        var ref2;
        return (ref2 = specificVersion != null ? specificVersion : latestVersion) != null ? ref2 : null;
    }
    /** Returns a boolean indicating if Expo Go should be installed. Returns `true` if the app was uninstalled. */ async uninstallExpoGoIfOutdatedAsync(deviceManager) {
        const cacheId = `${this.platform}-${deviceManager.identifier}`;
        if (ExpoGoInstaller.cache[cacheId]) {
            debug("skipping subsequent upgrade check");
            return false;
        }
        ExpoGoInstaller.cache[cacheId] = true;
        if (await this.isClientOutdatedAsync(deviceManager)) {
            if (this.sdkVersion === "UNVERSIONED") {
                // This should only happen in the expo/expo repo, e.g. `apps/test-suite`
                Log.log(`Skipping Expo Go upgrade check for UNVERSIONED project. Manually ensure the Expo Go app is built from source.`);
                return false;
            }
            // Only prompt once per device, per run.
            const confirm = await (0, _prompts).confirmAsync({
                initial: true,
                message: `Expo Go on ${deviceManager.name} is outdated, would you like to upgrade?`
            });
            if (confirm) {
                // Don't need to uninstall to update on iOS.
                if (this.platform !== "ios") {
                    Log.log(`Uninstalling Expo Go from ${this.platform} device ${deviceManager.name}.`);
                    await deviceManager.uninstallAppAsync(this.appId);
                }
                return true;
            }
        }
        return false;
    }
    /** Check if a given device has Expo Go installed, if not then download and install it. */ async ensureAsync(deviceManager) {
        let shouldInstall = !await deviceManager.isAppInstalledAsync(this.appId);
        if (_settings.APISettings.isOffline && !shouldInstall) {
            Log.warn(`Skipping Expo Go version validation in offline mode`);
            return false;
        } else if (_settings.APISettings.isOffline && shouldInstall) {
            throw new _errors.CommandError("NO_EXPO_GO", `Expo Go is not installed on device "${deviceManager.name}", while running in offline mode. Manually install Expo Go or run without --offline flag.`);
        }
        if (!shouldInstall) {
            shouldInstall = await this.uninstallExpoGoIfOutdatedAsync(deviceManager);
        }
        if (shouldInstall) {
            // Download the Expo Go app from the Expo servers.
            const binaryPath = await (0, _downloadExpoGoAsync).downloadExpoGoAsync(this.platform, {
                sdkVersion: this.sdkVersion
            });
            // Install the app on the device.
            const ora = (0, _ora).logNewSection(`Installing Expo Go on ${deviceManager.name}`);
            try {
                await deviceManager.installAppAsync(binaryPath);
            } finally{
                ora.stop();
            }
            return true;
        }
        return false;
    }
}
exports.ExpoGoInstaller = ExpoGoInstaller;

//# sourceMappingURL=ExpoGoInstaller.js.map